package mcfall.raytracer;

import java.awt.image.BufferedImage;
import java.awt.image.ImageConsumer;
import java.awt.image.MemoryImageSource;
import java.awt.image.WritableRaster;
import java.util.ArrayList;
import java.util.List;
import java.util.Observable;

import javax.swing.SwingUtilities;

import mcfall.math.Point;
import mcfall.math.Ray;
import mcfall.math.Vector;
import mcfall.raytracer.objects.HitRecord;
import mcfall.raytracer.objects.ProjectedExtent;
import mcfall.raytracer.objects.RGBIntensity;
import mcfall.raytracer.tests.ProjectionExtent;

import org.apache.log4j.Logger;

// TODO: remove dead code
/**
 * <p>
 * This class is the main engine for the ray tracer.  Calling the startProduction method will cause the RayTracer
 * to begin producing the pixels for the given view of the scene.  The location of the camera and the scene can
 * either be specified when the RayTracer object is constructed, or they can be specified before starting production
 * of the image.  If either of these properties is changed while a scene is being produced, the RayTracer will discard
 * the previously computed pixels of the scene and begin computing the new image.
 * </p>
 * <p>
 * Production of the image is done in a separate thread, so that the caller can continue with other tasks while the
 * image is being produced.  The RayTracer reports completion status of the scene through the usual ImageProducer/ImageConsumer
 * methods.
 * </p>
 * 
 * @author mcfall
 */
public class RayTracer extends Observable {
	/** The scene. */
	private Scene scene;	
	
	/** if shadows are enabled. */
	private boolean shadowsEnabled;
	
	/** The reflection enabled. */
	private boolean reflectionEnabled;
	
	private boolean transparencyEnabled;
	/** The image source. */
	private MemoryImageSource imageSource;

	private boolean boundingBoxEnabled;

	private boolean extrudeEnabled;

	private Integer pixelsRendered = 0; // this is an object Integer so we can do thread locking on it. 
	
	/** The logger. */
	private static Logger logger = Logger.getLogger("mcfall.raytracer");
	
	/**
	 * Constructs a RayTracer object that can trace the given scene from the point of view
	 * of a specific camera.  Shadows are enabled by default
	 * 
	 * @param scene the scene to be traced
	 */
	public RayTracer (Scene scene) {
		this.scene = scene;	
		setShadowsEnabled (true);
		setReflectionEnabled(true);
	}
	
	public WritableRaster startProduction(WritableRaster raster) 
	{
		return startProduction(raster,0,0,getCamera().getPlaneWidth(),getCamera().getPlaneHeight());
		
	}
	/**
	 * Start production of an image (or part of an image).
	 * 
	 * @param ic the ImageConsumer
	 * @param rowStart the starting row
	 * @param rowEnd the ending row + 1 (to allow getwidth functions to work)
	 * @param columnStart the start of a column
	 * @param columnEnd the the end of a column +1
	 * 
	 * @return the buffered image
	 */
	public WritableRaster startProduction(WritableRaster raster, int columnStart,int rowStart, int columnEnd, int rowEnd) {
		if(logger.isDebugEnabled()) {
			logger.debug("Starting image production:");
			logger.debug("Objects in scene:");
			for (ThreeDimensionalObject object : scene.getObjectList()) {
				logger.debug (object.getName());
			}
			logger.debug("Camera location:");
			logger.debug(getCamera());
		}
		if(raster==null) {
			BufferedImage image = new BufferedImage (getCamera().getPlaneWidth(), getCamera().getPlaneHeight(), java.awt.image.BufferedImage.TYPE_INT_ARGB);
			raster = image.getRaster();
		}
		int maxY = getCamera().getPlaneHeight ()-1;
		int[] color = new int[4];
		color[3] = 255;
		int localPixelsRendered = 0;
		for (int row = rowStart; row < rowEnd; row++) {
			for (int column = columnStart; column < columnEnd; column++) {
				
				Ray currentRay = getCamera().rayThrough(column, row);
				RGBIntensity colorValues = tracePixel(currentRay);
				
				color[0] = (int) (colorValues.getRed()*255);
				color[1] = (int) (colorValues.getGreen()*255);
				color[2] = (int) (colorValues.getBlue()*255);	
				if (logger.isDebugEnabled()) {
					logger.debug ("Testing ray through pixel (" + column + ", " + row + "): " + currentRay);
					logger.debug("Integer color values");
					logger.debug(color[0]);
					logger.debug(color[1]);
					logger.debug(color[2]);
				}
				raster.setPixel(column, maxY-row, color);
				localPixelsRendered++;
				if((localPixelsRendered)%10==0) { //every 20 pixels update the pixel count
					addPixelsRenderedSynched(localPixelsRendered);
					localPixelsRendered=0;
				}
			}						
		}
		addPixelsRenderedSynched(localPixelsRendered);
		return raster;
	}
	private void addPixelsRendered(int numRendered) {
		pixelsRendered+=numRendered;
		if(pixelsRendered>0 && 0==pixelsRendered%100){//every 100 pixels rendered in this render we shall notify the listeners 
			this.setChanged();
			this.notifyObservers();
		}
	}
	private void addPixelsRenderedSynched(int numRendered) {
		if(this.hasChanged()) {//avoid synching as often as possible in the RayTracer thread
			synchronized (pixelsRendered) {
				addPixelsRendered(numRendered);
			}
		} else {
			addPixelsRendered(numRendered);
		}
		
	}
	
	public int getPixelsDone() {
		synchronized(pixelsRendered) {
			this.clearChanged();
			return pixelsRendered;
		}
		
	}
	
	/**
	 * Determines the color of the object hit by the ray <i>currentRay</i>. 
	 * @param currentRay the current ray
	 * @return the RGB intensity of the object hit, or black if no object
	 * was hit
	 */
	private RGBIntensity tracePixel(Ray currentRay) {
		RGBIntensity black = new RGBIntensity(0, 0, 0);
		HitRecord hitRecord = scene.firstObjectHitBy(currentRay);
		if (hitRecord != null) {										
			if (logger.isDebugEnabled()) {
				logger.debug ("Hit object " + hitRecord.object.getName() + " at time " + hitRecord.hitTime + " at point " + currentRay.pointAt(hitRecord.hitTime));
			}
			RGBIntensity colorValues = computeColor (currentRay,hitRecord);
			return colorValues;
		}
		else {
			if (logger.isDebugEnabled()) {
				logger.debug (currentRay + " misses all objects");
			}
			return black;
		}
	}
	
	/**
	 * Compute color.
	 * 
	 * @param currentRay the current ray
	 * @param hitRecord the hit record
	 * 
	 * @return the RGB intensity
	 */
	private RGBIntensity computeColor(Ray currentRay, HitRecord hitRecord) {
		Point hitLocation = currentRay.pointAt(hitRecord.hitTime);
		Material material = hitRecord.object.getMaterial();
		Intensity red = new Intensity(0);
		Intensity green = new Intensity(0);
		Intensity blue = new Intensity(0);
		for (LightSource light : scene.getLights()) {						
			RGBIntensity specular = light.computeSpecular(getCamera().getLocation(), hitLocation, hitRecord.normal, 1);
			RGBIntensity diffuse = light.computeDiffuse(hitLocation, hitRecord.normal);

			if (logger.isDebugEnabled()) {
				logger.debug("Computing light contribution for light: " + light);
				logger.debug("Specular component is " + specular);
				logger.debug("Diffuse component is " + diffuse);
			}
			red.add(material.getSpecular().getRed()*specular.getRed());
			red.add(material.getDiffuse().getRed()*diffuse.getRed());
			
			green.add(material.getSpecular().getGreen()*specular.getGreen());
			green.add(material.getDiffuse().getGreen()*diffuse.getGreen());
			
			blue.add(material.getSpecular().getBlue()*specular.getBlue());
			blue.add(material.getDiffuse().getBlue()*diffuse.getBlue());
		}
		
		logger.debug ("Final color values:");
		logger.debug (red);
		logger.debug (green);
		logger.debug (blue);
		
		return new RGBIntensity (red.getValue(), green.getValue(), blue.getValue());
	}

	/**
	 * The main method.
	 * 
	 * @param args the arguments
	 * 
	 * @throws Exception the exception
	 */
	public static void main (String[] args) throws Exception {
		SwingUtilities.invokeLater( new Runnable () {
			
			/* (non-Javadoc)
			 * @see java.lang.Runnable#run()
			 */
			public void run () {
				AdvancedInfomation window = new TracerWindow ();
			}
		});		
	}

	/**
	 * Adds the consumer.
	 * 
	 * @param ic the ic
	 */
	public void addConsumer(ImageConsumer ic) {
		this.imageSource.addConsumer(ic);		
	}


	/**
	 * Checks to see if ic is a consumer.
	 * 
	 * @param ic the ic
	 * 
	 * @return true, if is consumer
	 */
	public boolean isConsumer(ImageConsumer ic) {
		return imageSource.isConsumer(ic);
	}


	/**
	 * Removes the consumer.
	 * @param ic the ic
	 */
	public void removeConsumer(ImageConsumer ic) {
		imageSource.removeConsumer(ic);
		
	}
	
	/**
	 * Gets the camera.
	 * 
	 * @return the camera
	 */
	private Camera getCamera() {
		return scene.getCamera ();
	}

	/**
	 * Checks if is shadows enabled.
	 * 
	 * @return true, if is shadows enabled
	 */
	public boolean isShadowsEnabled() {
		return shadowsEnabled;
	}

	/**
	 * Sets the shadows enabled.
	 * 
	 * @param shadowsEnabled the new shadows enabled
	 */
	public void setShadowsEnabled(boolean shadowsEnabled) {
		this.shadowsEnabled = shadowsEnabled;
	}

	/**
	 * Checks if is reflection enabled.
	 * 
	 * @return true, if is reflection enabled
	 */
	public boolean isReflectionEnabled() {
		return reflectionEnabled;
	}
	public boolean isTransparencyEnabled() {
		return transparencyEnabled;
	}
	public void setTransparencyEnabled(boolean transparencyEnabled) {
		this.transparencyEnabled=transparencyEnabled;
	}

	/**
	 * Sets the reflection enabled.
	 * 
	 * @param reflectionEnabled the new reflection enabled
	 */
	public void setReflectionEnabled(boolean reflectionEnabled) {
		this.reflectionEnabled = reflectionEnabled;
	}
	public void setBoundingBoxEnabled(boolean boundingBoxEnabled) {
		scene.setBoundingBoxEnabled(boundingBoxEnabled);
	}
	public boolean isBoundingBoxEnabled() {
		return scene.isBoundingBoxEnabled();
	}
	public void setExtrudeEnabled(boolean extrudeEnabled) {
		this.extrudeEnabled=extrudeEnabled;
	}
	public boolean isExtrudeEnabled() {
		return this.extrudeEnabled;
	}
	public int getImageWidth() 
	{
		return getCamera().getPlaneWidth();
	}
	public int getImageHeight() 
	{
		return getCamera().getPlaneHeight();
	}

}
